"use client";

import Image from "next/image";
import Link from "next/link";
import { usePathname } from "next/navigation";
import {
  IconLayoutDashboard,
  IconList,
  IconPlus,
  IconCalendar,
  IconStar,
  IconMessage,
  IconWallet,
  IconUser,
  IconFileText,
  IconReceipt,
  IconAlertCircle,
  IconFileCheck,
  IconShield,
  IconHelp,
  IconLogin,
  IconCurrencyDollar,
  IconUsers,
  IconCreditCard,
  IconPercentage,
  IconLogout,
  IconMenu2,
} from "@tabler/icons-react";
import {
  Box,
  Text,
  Stack,
  Divider,
  Flex,
  Drawer,
  ActionIcon,
} from "@mantine/core";
import { useMediaQuery } from "@mantine/hooks";
import theme from "../../styles/theme";
import Header from "@/shared/Header";
import { useState } from "react";

export default function Layout({ children }: { children: React.ReactNode }) {
  const pathname = usePathname();
  const [drawerOpened, setDrawerOpened] = useState(false);
  const isMobile = useMediaQuery("(max-width: 1200px)");

  const menuItems = [
    {
      title: "الرئيسية",
      items: [
        { label: "لوحة التحكم", icon: IconLayoutDashboard, path: "/dashboard" },
      ],
    },
    {
      title: "المساكن",
      items: [
        { label: "قائمة السكن", icon: IconList, path: "/properties" },
        { label: "إضافة سكن جديد", icon: IconPlus, path: "/properties/new" },
      ],
    },
    {
      title: "الحجوزات",
      items: [
        { label: "قائمة الحجوزات", icon: IconCalendar, path: "/bookings" },
      ],
    },
    {
      title: "التقويم",
      items: [{ label: "عرض التقويم", icon: IconCalendar, path: "/calendar" }],
    },
    {
      title: "التقييمات",
      items: [{ label: "قائمة التقييمات", icon: IconStar, path: "/reviews" }],
    },
    {
      title: "المحادثات",
      items: [{ label: "الرسائل", icon: IconMessage, path: "/messages" }],
    },
    {
      title: "المالية",
      items: [{ label: "الحسابات", icon: IconWallet, path: "/finance" }],
    },
    {
      title: "أخرى",
      items: [
        { label: "الملف الشخصي", icon: IconUser, path: "/profile" },
        {
          label: "ترخيص وزارة السياحة",
          icon: IconFileText,
          path: "/tourism-license",
        },
        { label: "الرقم الضريبي", icon: IconReceipt, path: "/tax-number" },
        { label: "البلاغات والشكاوي", icon: IconAlertCircle, path: "/reports" },
        { label: "اتفاقية العقد", icon: IconFileCheck, path: "/contract" },
        { label: "سياسة الخصوصية", icon: IconShield, path: "/privacy" },
        { label: "من نحن", icon: IconHelp, path: "/about" },
        {
          label: "برنامج حماية الضيف",
          icon: IconShield,
          path: "/guest-protection",
        },
        { label: "الدعم الفني", icon: IconHelp, path: "/support" },
      ],
    },
    {
      title: "الإعدادات",
      items: [
        { label: "الدخول والخروج", icon: IconLogin, path: "/auth" },
        { label: "الأسعار", icon: IconCurrencyDollar, path: "/pricing" },
        { label: "مسؤولي الحجز", icon: IconUsers, path: "/booking-managers" },
        { label: "الاشتراكات", icon: IconCreditCard, path: "/subscriptions" },
        { label: "الخصومات", icon: IconPercentage, path: "/discounts" },
        { label: "تسجيل الخروج", icon: IconLogout, path: "/logout" },
      ],
    },
  ];

  const SidebarContent = () => (
    <Stack align="center" gap="xs" py="xl">
      <Image
        src="/images/logo.png"
        alt="Katton Logo"
        width={100}
        height={50}
        style={{ marginBottom: "1rem" }}
      />
      <Text fw={700} size="xl" c={theme.colors.darkGray}>
        قطون
      </Text>
      <Text size="xs" c={theme.colors.darkGray}>
        استثمر معنا
      </Text>
    </Stack>
  );

  const MenuContent = () => (
    <Box px="md">
      {menuItems.map((section, index) => (
        <Box key={index} mb="lg">
          <Text fw={600} size="xs" mb="sm" c={theme.colors.navy}>
            {section.title}
          </Text>
          <Stack gap="xs">
            {section.items.map((item, itemIndex) => {
              const isActive = pathname === item.path;
              const Icon = item.icon;

              return (
                <Link
                  key={itemIndex}
                  href={item.path}
                  style={{
                    textDecoration: "none",
                    color: "inherit",
                  }}
                  onClick={() => isMobile && setDrawerOpened(false)}
                >
                  <Box
                    style={{
                      gap: 20,
                      display: "flex",
                      alignItems: "center",
                      padding: "0.5rem",
                      borderRadius: "0.5rem",
                      backgroundColor: isActive ? "#f9fafb" : "transparent",
                      "&:hover": {
                        backgroundColor: "#f9fafb",
                      },
                    }}
                  >
                    <Icon
                      size={18}
                      color={isActive ? theme.colors.gold : theme.colors.black}
                    />
                    <Text
                      size="sm"
                      fw={isActive ? 500 : 400}
                      c={isActive ? theme.colors.gold : theme.colors.black}
                    >
                      {item.label}
                    </Text>
                  </Box>
                </Link>
              );
            })}
          </Stack>
          {index < menuItems.length - 1 && <Divider my="md" />}
        </Box>
      ))}
    </Box>
  );

  return (
    <Flex>
      {!isMobile && (
        <Box
          component="nav"
          style={{
            position: isMobile ? "relative" : "fixed",
            right: 0,
            top: 0,
            height: "100vh",
            width: isMobile ? "100%" : "250px",
            backgroundColor: "white",
            borderLeft: "1px solid #e5e7eb",
            overflowY: "auto",
          }}
        >
          <SidebarContent />
          <MenuContent />
        </Box>
      )}

      {isMobile && (
        <Drawer
          opened={drawerOpened}
          onClose={() => setDrawerOpened(false)}
          position="left"
          size="250px"
          withCloseButton={false}
        >
          <SidebarContent />
          <MenuContent />
        </Drawer>
      )}

      <Box
        style={{
          marginRight: isMobile ? 0 : "250px",
          width: isMobile ? "100%" : "calc(100% - 250px)",
          minHeight: "100vh",
        }}
      >
        <div style={{ display: "flex", alignItems: "center" }}>
          {isMobile && (
            <ActionIcon
              variant="transparent"
              onClick={() => setDrawerOpened(true)}
              style={{ marginRight: "1rem" }}
            >
              <IconMenu2 size={24} color={theme.colors.darkGray} />
            </ActionIcon>
          )}
          <Header />
        </div>
        <Box style={{ paddingTop: "70px" }}>{children}</Box>
      </Box>
    </Flex>
  );
}
