import { Modal, Text, Button, Group, Box, Flex } from "@mantine/core";
import { IconCheck } from "@tabler/icons-react";
import theme from "@/styles/theme";
import { useState } from "react";

interface AddPropertyModalProps {
  opened: boolean;
  onClose: () => void;
}

interface StepStatus {
  isCompleted: boolean;
  isLoading: boolean;
}

export function AddPropertyModal({ opened, onClose }: AddPropertyModalProps) {
  const [contractStep, setContractStep] = useState<StepStatus>({
    isCompleted: false,
    isLoading: false,
  });
  const [licenseStep, setLicenseStep] = useState<StepStatus>({
    isCompleted: false,
    isLoading: false,
  });
  const [profileStep, setProfileStep] = useState<StepStatus>({
    isCompleted: false,
    isLoading: false,
  });
  const [isSubmitting, setIsSubmitting] = useState(false);

  const handleContractSubmit = async () => {
    setContractStep({ ...contractStep, isLoading: true });
    try {
      // محاكاة طلب API
      await new Promise((resolve) => setTimeout(resolve, 1500));
      setContractStep({ isCompleted: true, isLoading: false });
    } catch (error) {
      console.error("Contract submission error:", error);
      setContractStep({ isCompleted: false, isLoading: false });
    }
  };

  const handleLicenseSubmit = async () => {
    setLicenseStep({ ...licenseStep, isLoading: true });
    try {
      await new Promise((resolve) => setTimeout(resolve, 1500));
      setLicenseStep({ isCompleted: true, isLoading: false });
    } catch (error) {
      console.error("License submission error:", error);
      setLicenseStep({ isCompleted: false, isLoading: false });
    }
  };

  const handleProfileSubmit = async () => {
    setProfileStep({ ...profileStep, isLoading: true });
    try {
      await new Promise((resolve) => setTimeout(resolve, 1500));
      setProfileStep({ isCompleted: true, isLoading: false });
    } catch (error) {
      console.error("Profile submission error:", error);
      setProfileStep({ isCompleted: false, isLoading: false });
    }
  };

  const handleFinalSubmit = async () => {
    if (
      !contractStep.isCompleted ||
      !licenseStep.isCompleted ||
      !profileStep.isCompleted
    ) {
      return;
    }

    setIsSubmitting(true);
    try {
      await new Promise((resolve) => setTimeout(resolve, 2000));
      onClose();
    } catch (error) {
      console.error("Final submission error:", error);
    } finally {
      setIsSubmitting(false);
    }
  };

  return (
    <Modal
      opened={opened}
      onClose={onClose}
      centered
      size="lg"
      styles={{
        content: {
          borderRadius: "20px",
          background: "#FFF",
          boxShadow: "8px 8px 108px 0px rgba(0, 0, 0, 0.05)",
        },
      }}
    >
      <Box ta="center" mb="xl">
        <Text fw={700} size="xl" mb="xs">
          تنبيه
        </Text>
        <Text c="#999">يرجي اكمال البيانات التالية لاضافة مسكن</Text>
      </Box>

      <Box>
        <Group justify="flex-start" mb="md">
          <Box
            style={{
              backgroundColor: contractStep.isCompleted
                ? "#07901C15"
                : "#8C808215",
              padding: "8px",
              borderRadius: "50%",
              display: "flex",
              alignItems: "center",
              justifyContent: "center",
            }}
          >
            <IconCheck
              size={20}
              color={contractStep.isCompleted ? "#07901C" : "#8C8082"}
            />
          </Box>
          <Text>
            الموافقة علي{" "}
            <Text span c={theme.colors.gold}>
              العقد
            </Text>
          </Text>
        </Group>
        <Button
          radius="md"
          mb="xl"
          bg={theme.colors.gold}
          loading={contractStep.isLoading}
          onClick={handleContractSubmit}
          disabled={contractStep.isCompleted}
        >
          {contractStep.isCompleted ? "تمت الموافقة" : "تصفح العقد والموافقة"}
        </Button>

        <Group justify="flex-start" mb="md">
          <Box
            style={{
              backgroundColor: licenseStep.isCompleted
                ? "#07901C15"
                : "#8C808215",
              padding: "8px",
              borderRadius: "50%",
              display: "flex",
              alignItems: "center",
              justifyContent: "center",
            }}
          >
            <IconCheck
              size={20}
              color={licenseStep.isCompleted ? "#07901C" : "#8C8082"}
            />
          </Box>
          <Text>اضافة ترخيص وزارة السياحة</Text>
        </Group>
        <Button
          mb="xl"
          radius="md"
          bg={theme.colors.gold}
          loading={licenseStep.isLoading}
          onClick={handleLicenseSubmit}
          disabled={licenseStep.isCompleted}
        >
          {licenseStep.isCompleted ? "تم إضافة الترخيص" : "اضافة ترخيص جديد"}
        </Button>

        <Group justify="flex-start" mb="md">
          <Box
            style={{
              backgroundColor: profileStep.isCompleted
                ? "#07901C15"
                : "#8C808215",
              padding: "8px",
              borderRadius: "50%",
              display: "flex",
              alignItems: "center",
              justifyContent: "center",
            }}
          >
            <IconCheck
              size={20}
              color={profileStep.isCompleted ? "#07901C" : "#8C8082"}
            />
          </Box>
          <Text>اكمال بيانات الملف الشخصي</Text>
        </Group>
        <Button
          mb="xl"
          radius="md"
          bg={theme.colors.gold}
          loading={profileStep.isLoading}
          onClick={handleProfileSubmit}
          disabled={profileStep.isCompleted}
        >
          {profileStep.isCompleted ? "تم إكمال البيانات" : "الملف الشخصي"}
        </Button>

        <Flex justify="center">
          <Button
            radius="md"
            bg="#8C8082"
            w={{ base: "100%", md: "30%" }}
            onClick={handleFinalSubmit}
            loading={isSubmitting}
            disabled={
              !contractStep.isCompleted ||
              !licenseStep.isCompleted ||
              !profileStep.isCompleted
            }
          >
            تم
          </Button>
        </Flex>
      </Box>
    </Modal>
  );
}
